
// =====================
// VALIDAZIONE LIQUIDAZIONE SPESE (clean)
// =====================

// Regex numeri con , o . e massimo 2 decimali
const NUM_RE = /^\d+(?:[.,]\d{1,2})?$/;

// Parser numeri: accetta "12,34" o "12.34"
function _parseNum(v) {
  v = (v ?? '').toString().trim().replace(',', '.');
  const n = Number(v);
  return Number.isFinite(n) ? n : 0;
}
function _fmt2(n) { return Number(n || 0).toFixed(2); }

// ---------------------
// A) HANDLER CAMPI TABELLA (allineati ai nomi dell’HTML)
// ---------------------

$(document).on("change", "select[name=categoria_interventi]", function () {
  valida_campo($(this), '.error_categoria_interventi', null, null, "Seleziona la categoria di intervento.");
});

$(document).on("input", "input[name=n_fattura]", function () {
  valida_campo($(this), '.error_n_fattura', null, null, "Inserisci il numero della fattura.");
});

$(document).on("input", "input[name=emittente_fattura]", function () {
  valida_campo($(this), '.error_emittente_fattura', null, null, "Inserisci l'emittente della fattura.");
});

$(document).on("input", "input[name=descrizione_fattura]", function () {
  valida_campo($(this), '.error_descrizione_fattura', null, null, "Inserisci la descrizione.");
});

// Importi: validazione + ricalcolo
$(document).on("input", "input[name=importo_imponibile]", function () {
  valida_campo($(this), '.error_importo_imponibile', NUM_RE, "Importo non valido.", "Inserisci l'importo.");
  if (!$(this).next('.error_importo_imponibile').hasClass("invalid-feedback")) calcola_totale_liquidazione();
});

$(document).on("input", "input[name=importo_ivato]", function () {
  valida_campo($(this), '.error_importo_ivato', NUM_RE, "Importo non valido.", "Inserisci l'importo.");
  if (!$(this).next('.error_importo_ivato').hasClass("invalid-feedback")) calcola_totale_liquidazione();
});

$(document).on("input", "input[name=contributo_richiesto]", function () {
  valida_campo($(this), '.error_contributo_richiesto', NUM_RE, "Importo non valido.", "Inserisci l'importo.");
  if (!$(this).next('.error_contributo_richiesto').hasClass("invalid-feedback")) calcola_totale_liquidazione();
});

// Copia fattura (PDF)
$(document).on("change", "input[name=copia_fattura]", function () {
  valida_campo($(this), '.error_copia_fattura', null, null, "Documento fattura obbligatorio.");
});

// ---------------------
// B) VALIDAZIONE DRS (live, non bloccante)
// ---------------------

// DRS n.
$("#numeroDRS").on("input", function () {
  const val = $(this).val().trim();
  const regexDRS = /^\d+(?:\/\d+)?$/; // 1234 oppure 1234/2024
  if (val === "") { $(this).removeClass("is-invalid"); $("#errore_numeroDRS").text(""); return; }
  if (!regexDRS.test(val)) {
    $(this).addClass("is-invalid");
    $("#errore_numeroDRS").addClass("invalid-feedback").text("Il numero DRS deve contenere solo numeri (es. 1234).");
  } else {
    $(this).removeClass("is-invalid");
    $("#errore_numeroDRS").text("");
  }
});

// Data DRS
$("#dataDRS").on("input", function () {
  if (!$(this).val()) {
    $(this).addClass("is-invalid");
    $("#errore_dataDRS").addClass("invalid-feedback").text("Inserisci la data del DRS.");
  } else {
    $(this).removeClass("is-invalid");
    $("#errore_dataDRS").text("");
  }
});

// Importo Ammissibile
$("#importoAmmissibile").on("input", function () {
  const val = $(this).val().trim();
  if (val === "" || !NUM_RE.test(val)) {
    $(this).addClass("is-invalid");
    $("#errore_importoAmmissibile").addClass("invalid-feedback").text("Inserisci un importo valido.");
  } else {
    $(this).removeClass("is-invalid");
    $("#errore_importoAmmissibile").text("");
  }
  calcola_totale_liquidazione();
});

function calcola_totale_liquidazione() {
  let sImp = 0, sIvato = 0, sCtr = 0;

  // Somme (ignora input marcati invalid)
  $("#tbody-investimenti input.imponibile").each(function () {
    if (!$(this).hasClass('is-invalid')) sImp += _parseNum($(this).val());
  });
  $("#tbody-investimenti input.ivato").each(function () {
    if (!$(this).hasClass('is-invalid')) sIvato += _parseNum($(this).val());
  });
  $("#tbody-investimenti input.contributo").each(function () {
    if (!$(this).hasClass('is-invalid')) sCtr += _parseNum($(this).val());
  });

  // Aggiorna i totali
  $("#importo_imponibile_totale").text(_fmt2(sImp));
  $("#importo_imponibile_iva_totale").text(_fmt2(sIvato));
  $("#contributo_richiesto_totale").text(_fmt2(sCtr));

  // Alert #1: contributo richiesto > importo massimo concesso
  const contributoMassimo = _parseNum($("#importoAmmissibile").val());
  const $alert1 = $("#alert_contributo");
  if (contributoMassimo > 0 && sCtr > contributoMassimo) $alert1.show(); else $alert1.hide();

  // Beneficiario
  const tipoBeneficiario = ($("input[name='tipo_beneficiario']:checked").val() || "").toLowerCase();
  const isComune = (tipoBeneficiario === "comuni");

  // Alert #2 (NON-Comuni): Ivato >= Contributo * 1.25
  const $alert2 = $("#alert_ivato_vs_contributo");
  if (!isComune && sCtr > 0 && sIvato < sCtr * 1.25) $alert2.show(); else $alert2.hide();

  // Alert #3 (SOLO Comuni): Ivato >= Contributo richiesto
  const $alert3 = $("#alert_comuni_ivato_vs_contributo");
  if (isComune && sCtr > 0 && sIvato < sCtr) $alert3.show(); else $alert3.hide();
}

// Ricalcolo automatico
$("input[name='tipo_beneficiario']").on("change", calcola_totale_liquidazione);
$("#tbody-investimenti").on("input", "input.imponibile, input.ivato, input.contributo", calcola_totale_liquidazione);
// D) FUNZIONE DI CHECK (per uso esterno)
// ---------------------
function check_sezione_liquidazione_spese() {
  console.clear(); // pulizia console a ogni validazione
  console.group("🔍 Validazione Sezione Liquidazione Spese");

  // forza validazioni live già collegate
  $("#numeroDRS").trigger("input");
  $("#dataDRS").trigger("input");
  $("#importoAmmissibile").trigger("input");

  // pulizia stato precedente
  let err = 0;
  let $firstInvalid = null;
  let errorList = [];

  $("#tbody-investimenti input, #numeroDRS, #dataDRS, #importoAmmissibile")
    .removeClass("is-invalid");
  $("#tbody-investimenti .error_categoria_interventi, \
     #tbody-investimenti .error_n_fattura, \
     #tbody-investimenti .error_emittente_fattura, \
     #tbody-investimenti .error_descrizione_fattura, \
     #tbody-investimenti .error_importo_imponibile, \
     #tbody-investimenti .error_importo_ivato, \
     #tbody-investimenti .error_contributo_richiesto, \
     #tbody-investimenti .error_copia_fattura").text("");
  $("#tbody-investimenti tr").removeClass("riga-errore");

  const NUM_RE = /^\d+(?:[.,]\d{1,2})?$/;
  const regexDRS = /^\d+(?:\/\d+)?$/;

  // --- DRS n. ---
  if (!regexDRS.test(($("#numeroDRS").val() || "").trim())) {
    $("#numeroDRS").addClass("is-invalid");
    $("#errore_numeroDRS").text("Inserisci un numero DRS valido (es. 1234).");
    $firstInvalid = $firstInvalid || $("#numeroDRS");
    errorList.push({ campo: "Numero DRS", valore: $("#numeroDRS").val(), messaggio: "Formato non valido" });
    err++;
  }

  // --- Data DRS ---
  if (!$("#dataDRS").val()) {
    $("#dataDRS").addClass("is-invalid");
    $("#errore_dataDRS").text("Inserisci la data del DRS.");
    $firstInvalid = $firstInvalid || $("#dataDRS");
    errorList.push({ campo: "Data DRS", valore: $("#dataDRS").val(), messaggio: "Campo obbligatorio" });
    err++;
  }

  // --- Importo ammissibile ---
  const valImp = ($("#importoAmmissibile").val() || "").trim();
  if (valImp === "" || !NUM_RE.test(valImp)) {
    $("#importoAmmissibile").addClass("is-invalid");
    $("#errore_importoAmmissibile").text("Inserisci un importo valido.");
    $firstInvalid = $firstInvalid || $("#importoAmmissibile");
    errorList.push({ campo: "Importo Ammissibile", valore: valImp, messaggio: "Importo non valido" });
    err++;
  }

  // --- Righe tabella ---
  $("#tbody-investimenti tr").each(function (i) {
    const riga = i + 1;
    const $row = $(this);
    let rowErrors = 0;

    const $categoria = $row.find("select[name=categoria_interventi]");
    const $nFattura = $row.find("input[name=n_fattura]");
    const $emittente = $row.find("input[name=emittente_fattura]");
    const $descr = $row.find("input[name=descrizione_fattura]");
    const $imponibile = $row.find("input[name=importo_imponibile]");
    const $ivato = $row.find("input[name=importo_ivato]");
    const $contrib = $row.find("input[name=contributo_richiesto]");
    const $pdf = $row.find("input[name=copia_fattura]");

    // Categoria
    if (!$categoria.val()) {
      $categoria.addClass("is-invalid");
      //  $row.find(".error_categoria_interventi").text("Seleziona la categoria.");
      $firstInvalid = $firstInvalid || $categoria;
      errorList.push({ campo: `Riga ${riga} - Categoria`, valore: "", messaggio: "Categoria obbligatoria" });
      rowErrors++;
    }

    // Numero fattura
    if (!($nFattura.val() || "").trim()) {
      $nFattura.addClass("is-invalid");
      // $row.find(".error_n_fattura").text("Inserisci il numero della fattura.");
      $firstInvalid = $firstInvalid || $nFattura;
      errorList.push({ campo: `Riga ${riga} - Numero Fattura`, valore: $nFattura.val(), messaggio: "Campo obbligatorio" });
      rowErrors++;
    }

    // Emittente fattura
    if (!($emittente.val() || "").trim()) {
      $emittente.addClass("is-invalid");
      //$row.find(".error_emittente_fattura").text("Inserisci l'emittente.");
      $firstInvalid = $firstInvalid || $emittente;
      errorList.push({ campo: `Riga ${riga} - Emittente`, valore: $emittente.val(), messaggio: "Campo obbligatorio" });
      rowErrors++;
    }

    // Descrizione fattura
    if (!($descr.val() || "").trim()) {
      $descr.addClass("is-invalid");
      // $row.find(".error_descrizione_fattura").text("Inserisci la descrizione.");
      $firstInvalid = $firstInvalid || $descr;
      errorList.push({ campo: `Riga ${riga} - Descrizione`, valore: $descr.val(), messaggio: "Campo obbligatorio" });
      rowErrors++;
    }

    // Importi
    if (!NUM_RE.test(($imponibile.val() || "").trim())) {
      $imponibile.addClass("is-invalid");
      // $row.find(".error_importo_imponibile").text("Importo non valido.");
      $firstInvalid = $firstInvalid || $imponibile;
      errorList.push({ campo: `Riga ${riga} - Imponibile`, valore: $imponibile.val(), messaggio: "Importo non valido" });
      rowErrors++;
    }
    if (!NUM_RE.test(($ivato.val() || "").trim())) {
      $ivato.addClass("is-invalid");
      // $row.find(".error_importo_ivato").text("Importo non valido.");
      $firstInvalid = $firstInvalid || $ivato;
      errorList.push({ campo: `Riga ${riga} - Ivato`, valore: $ivato.val(), messaggio: "Importo non valido" });
      rowErrors++;
    }
    if (!NUM_RE.test(($contrib.val() || "").trim())) {
      $contrib.addClass("is-invalid");
      // $row.find(".error_contributo_richiesto").text("Importo non valido.");
      $firstInvalid = $firstInvalid || $contrib;
      errorList.push({ campo: `Riga ${riga} - Contributo`, valore: $contrib.val(), messaggio: "Importo non valido" });
      rowErrors++;
    }

    // PDF obbligatorio
    if ($pdf[0]?.files.length !== 1) {
      $pdf.addClass("is-invalid");
      $row.find(".error_copia_fattura").text("Carica il file PDF.");
      $firstInvalid = $firstInvalid || $pdf;
      errorList.push({ campo: `Riga ${riga} - PDF`, valore: "", messaggio: "File obbligatorio" });
      rowErrors++;
    }

    if (rowErrors > 0) $row.addClass("riga-errore");
    err += rowErrors;
  });

  // 🔴 Conta gli alert attivi nella sezione Spese (anche se l’accordion è chiuso)
  let alertCount = 0;
  const $alertContrib = $("#alert_contributo");
  const $alertIvato = $("#alert_ivato_vs_contributo");
  const $alertComune = $("#alert_comuni_ivato_vs_contributo"); // <--- NUOVO

  // controlla se sono visibili o comunque attivi
  if ($alertContrib.length && $alertContrib.css("display") !== "none") {
    alertCount++;
    errorList.push({
      campo: "Sezione Spese",
      valore: "Alert contributo richiesto",
      messaggio: "Contributo richiesto superiore all’importo massimo concesso"
    });
  }
  if ($alertIvato.length && $alertIvato.css("display") !== "none") {
    alertCount++;
    errorList.push({
      campo: "Sezione Spese",
      valore: "Alert ivato vs contributo",
      messaggio: "Importo ivato non coerente con il contributo richiesto"
    });
  }

  // <<< NUOVO blocco per i Comuni
if ($alertComune.length && $alertComune.css("display") !== "none") {
  alertCount++;
  errorList.push({
    campo: "Sezione Spese",
    valore: "Alert Comuni ivato vs contributo",
    messaggio: "Per i Comuni l'importo ivato deve essere ≥ contributo richiesto"
  });
}
  if (alertCount > 0) {
    err += alertCount; // aumenta il numero di errori
    console.warn(`⚠️ Trovati ${alertCount} alert in sezione spese`);
    console.table(errorList);

    // se non ci sono altri campi invalidi, scorri fino al primo alert
    if (!$firstInvalid) {
      const $firstAlert = [$alertContrib, $alertIvato]
        .find($el => $el && $el.length && $el.css("display") !== "none");
      if ($firstAlert && $firstAlert.length)
        $('html, body').animate({ scrollTop: $firstAlert.offset().top - 120 }, 350);
    }
  }
  // badge
  const $badge = $("#validazione_sezione_liquidazione_spese");
  const $btn = $("#liquidazione-spese .accordion-header button");
  $btn.removeClass("background-color-success-accordion background-color-error-accordion");
  $badge.removeClass("badge text-bg-success text-bg-error");

  if (err === 0) {
    $badge.text("Sezione Compilata Correttamente").addClass("badge text-bg-success");
    $btn.addClass("background-color-success-accordion");
    console.info("✅ Nessun errore trovato");
  } else {
    $badge.text(`Vi ${err === 1 ? "è" : "sono"} ${err} ${err === 1 ? "errore" : "errori"}`)
      .addClass("badge text-bg-error");
    $btn.addClass("background-color-error-accordion");
    console.warn(`⚠️ Totale errori: ${err}`);
    console.table(errorList);
  }

  // scroll al primo errore
  if ($firstInvalid && $firstInvalid.length) {
    $('html, body').animate({ scrollTop: $firstInvalid.offset().top - 120 }, 350);
    $firstInvalid.trigger("focus");
  }

  console.groupEnd();
  return err;
}
// Debug: stampa tutti i campi invalid
/*
(function debugInvalid() {
  const invalid = [];
  $("#numeroDRS, #dataDRS, #importoAmmissibile, #tbody-investimenti input, #tbody-investimenti select")
    .each(function () {
      if ($(this).hasClass("is-invalid")) {
        invalid.push({
          name: this.name || this.id,
          value: $(this).val(),
          row: $(this).closest("tr").attr("id") || "(no-row)"
        });
      }
    });
  console.group("LIQUIDAZIONE – campi invalid");
  console.table(invalid);
  console.groupEnd();
})();*/
// ---------------------
// E) INIT
// ---------------------
$(function () {
  // Imposta attributi numerici sui campi delle righe presenti
  $("#tbody-investimenti").find(".imponibile, .ivato, .contributo").attr({
    type: "number", step: "0.01", min: "0", inputmode: "decimal"
  });
  // Limitazione PDF per la copia fattura
  const cf = $("#tbody-investimenti").find('input[name="copia_fattura"]').get(0);
  if (cf) cf.setAttribute('accept', 'application/pdf');

  calcola_totale_liquidazione();
});

// Aggiungi riga "Liquidazione delle spese"
$(document).on("click", ".aggiungi_riga_investimenti", function (e) {
  e.preventDefault();

  const $tbody = $("#tbody-investimenti");
  const $tmpl = $tbody.find("tr").first();         // riga modello
  const nextIdx = $tbody.find("tr").length + 1;

  // clona la riga e pulisci i campi
  const $row = $tmpl.clone(false, false);
  $row.attr("id", "categoria_interventi_" + nextIdx)
    .attr("data-errori-riga", "0")
    .removeClass("riga-errore");

  // reset campi testo/select
  $row.find('select[name="categoria_interventi"]').val("").removeClass("is-invalid");
  $row.find('input[name="n_fattura"]').val("").removeClass("is-invalid");
  $row.find('input[name="emittente_fattura"]').val("").removeClass("is-invalid");
  $row.find('input[name="descrizione_fattura"]').val("").removeClass("is-invalid");

  // reset importi (imponibile/ivato/contributo)
  $row.find('input[name="importo_imponibile"]').val("").removeClass("is-invalid");
  $row.find('input[name="importo_ivato"]').val("").removeClass("is-invalid");
  $row.find('input[name="contributo_richiesto"]').val("").removeClass("is-invalid");

  // reset messaggi errore inline
  $row.find(".error_categoria_interventi, .error_n_fattura, .error_emittente_fattura, .error_descrizione_fattura, .error_importo_imponibile, .error_importo_ivato, .error_contributo_richiesto, .error_copia_fattura")
    .text("");
  $row.find(".invalid-feedback").text(""); // quelle sotto gli input numerici

  // file PDF: svuota, mantieni solo name e accept; niente id duplicati
  const $file = $row.find('input[name="copia_fattura"]');
  $file.val("").removeClass("is-invalid").removeAttr("id").attr("accept", "application/pdf");

  // inserisci in tabella
  $tbody.append($row);

  // aggiorna contatore righe e ricalcola i totali
  $("#count_righe_tabella_investimenti").val(nextIdx);
  if (typeof calcola_totale_liquidazione === "function") {
    calcola_totale_liquidazione();
  }

  // scroll dolce alla nuova riga
  $('html, body').animate({ scrollTop: $row.offset().top - 120 }, 300);
});



// Clic sul cestino -> elimina la riga (o la resetta se è l’unica)
$(document).on('click', '.elimina-riga-investimenti', function () {
  const $tr = $(this).closest('tr');
  const $tbody = $('#tbody-investimenti');
  const righe = $tbody.find('tr').length;

  // conferma (facoltativa)
  if (!confirm('Vuoi eliminare questa riga?')) return;

  if (righe <= 1) {
    // Non possiamo rimuovere l’ultima riga: la resettiamo
    resetRigaInvestimenti($tr);
  } else {
    $tr.remove();
    // opzionale: rinumerare gli id riga
    $tbody.find('tr').each(function (idx) {
      $(this).attr('id', 'categoria_interventi_' + (idx + 1));
    });
    // opzionale: aggiornare il contatore “righe”
    $('#count_righe_tabella_investimenti').val($tbody.find('tr').length);
  }

  // ricalcolo totali e badge se li usi
  if (typeof calcola_totale_liquidazione === 'function') {
    calcola_totale_liquidazione();
  }
  if (typeof check_sezione_liquidazione_spese === 'function') {
    // non obbligatorio; se lo vuoi, scommenta:
    // check_sezione_liquidazione_spese();
  }
});